/* vim: set sw=8 ts=8 si : */
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * See http://www.gnu.org/copyleft/ for details.
 *
 * Written by Guido Socher <guido at linuxfocus.org>
 *
 */
#include <avr/io.h>
#include <math.h>
#include "ntc.h"


/* this function converts the ADC value into an Ohm value of the NTC
 * here AVcc=5V is used as reference voltage 
 */
double adc_2_ohm_avcc(int adc_value){
	// 10bit adc=0..1023 over a voltage range from 0-5V
	// Uref=5V = AVcc
	// 10000 / ( 1024/adc_value  - 1) -> ohm
	// 
	// +5V
	// ---+
	//    #
	//    # 10k constant resistor
	//    |---------> Uadc 
	//    #
	//    # ntc 4.7k
	// ---+
	// 
	// ADCval= Uadc * (1024/Uref)
	// Rntc= 10K * ( 1 / ((5V/Uadc) -1) )
	if (adc_value < 1){
		// never divide by zero:
		adc_value=1;
	}
	double ohm;
	ohm=10000.0 / (( 1024.0 / adc_value ) - 1 );
	return(ohm);
}

/* this function converts the ADC value into an Ohm value of the NTC
 * here the Atmega8 internal 2.56V is used as reference voltage 
 */
double adc_2_ohm_iref(int adc_value){
	// 10bit adc=0..1023 over a voltage range from 0-2.56V
	// Uref=2.56V
	// 10000 / ( 2000/adc_value  - 1) -> ohm
	// 
	// +5V
	// ---+
	//    #
	//    # 10k constant resistor
	//    |---------> Uadc 
	//    #
	//    # ntc 4.7k
	// ---+
	// 
	// ADCval= Uadc * (1024/Uref)
	// Rntc= 10K * ( 1 / ((5V/Uadc) -1) )
	if (adc_value < 1){
		// never divide by zero:
		adc_value=1;
	}
	double ohm;
	ohm=10000.0 / (( 2000.0 / adc_value ) - 1 );
	return(ohm);
}

/* convert a ntc resistance value given in kohm to
 * temperature in celsius. The NTC follows a exponential
 * characteristic. */
double r2temperature(double ohm,int inout)
{
	double tcelsius=100;
	double tmp=1;
	if ((ohm/NTC_RNi) < 0.1){
		//ERROR: resistance value of NTC is too small (too hot here ;-)
		return(tcelsius);
	}
	if (inout){
		tmp=((1.0/NTC_Bi)*log(ohm/NTC_RNi)) + (1/(NTC_TNi+273.0));
	}else{
		tmp=((1.0/NTC_Bo)*log(ohm/NTC_RNo)) + (1/(NTC_TNo+273.0));
	}
	tcelsius=(1.0/tmp) -273;
	return(tcelsius);
}

